/******************************************************************************
 Copyright (C), 2014, TP-LINK TECHNOLOGIES CO., LTD.  All rights reserved.

 File name	: wlanIoctl.c
 Version	: v1.0
 Description: get wlan driver running info from BCM driver.

 Author		: Jiangji
 Create Date: 2014/02/17

 History: 
------------------------------------------------------------
 01, 17Feb14, jiangji, create this file.
***********************************************************/
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <net/if.h>			/* for struct if_req*/
#include <arpa/inet.h>		/* IPPROTORAW */
#include <linux/ethtool.h>	/* for ETHTOOL_GDRVINFO */
#include <linux/sockios.h>	/* for SIOCETHTOOL */
#include <errno.h>			/* fro errno */

#include "wlanIoctl.h"


#define WLC_GET_MAGIC					0
#define WLC_GET_VERSION					1
#define WLC_GET_BSSID					23
#define WLC_GET_ASSOCLIST				159
#define WLC_GET_VAR						262

#define WLC_IOCTL_VERSION					2
#define WLC_IOCTL_VERSION_LEGACY_IOTYPES	1
#define WLC_IOCTL_MAGIC						0x14e46c77

typedef struct wl_ioctl {
	unsigned int 	cmd;		/* common ioctl definition */
	void 			*buf;		/* pointer to user buffer */
	unsigned int	len;		/* length of user buffer */
	unsigned char	set;		/* get or set request (optional) */
	unsigned int	used;		/* bytes read or written (optional) */
	unsigned int	needed;		/* bytes needed (optional) */
} wl_ioctl_t;

/* dig into driver and load info to @buf */
static int wlanIoctl(const char *ifName, int cmd, void *buf, int len)
{
	struct ifreq ifr;
	wl_ioctl_t ioc;
	int ret = 0;
	int s = 0;

	/* create socket to kernel */
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
	{
		perror("wlanWlIoctl:socket");
		return errno;
	}

	memset(&ifr, '\0', sizeof(ifr));

	/* do a ioctl */
	ioc.cmd = cmd;
	ioc.buf = buf;
	ioc.len = len;

	/* init remaining fields */
	ioc.set = (unsigned char)0;

	strncpy(ifr.ifr_name, ifName, IFNAMSIZ);
	ifr.ifr_data = (caddr_t)&ioc;

	if ((ret = ioctl(s, SIOCDEVPRIVATE, &ifr)) < 0)
	{
		if ((cmd != WLC_GET_MAGIC) && (cmd != WLC_GET_BSSID))
		{
			perror(ifr.ifr_name);
		}
	}

	close(s);
	return ret;
}

/* get int var's value from wlan driver */
static int wl_iovar_getint(const char *ifName, const char *varName, int *val)
{
	int ret = -1;
	int nameLen = 0;
	char buf[100] = {'\0'};

	memset(buf, 0, 100);
	nameLen = strlen(varName) + 1;
	memcpy(buf, varName, nameLen);

	ret = wlanIoctl(ifName, WLC_GET_VAR, buf, sizeof(buf));
	if (ret < 0)
	{
		return (-1);
	}

	*val = *(int*)buf;

	return 0;
}


/******************
 * Public functions
 ******************/


/* check wlan interface named by @ifName exist 
 * 0 for exist.
 */
int wl_check(const char *ifName)
{
	int ret = -1;
	int val = 0;

	/* printf("%s: %s.\r\n", __func__, ifName); */

	/* ethx/wlx.x */
	if ((ifName == NULL && strlen(ifName) >= 2) ||
		(strncmp(ifName, "wl", 2) && strncmp(ifName, "eth", 3)))
	{
		return (-1);
	}

	/* check magic */
	ret = wlanIoctl(ifName, WLC_GET_MAGIC, &val, sizeof(int));
	if (ret < 0)
	{
		return ret;
	}

	if (val != WLC_IOCTL_MAGIC)
	{
		return (-1);
	}

	/* check version */
	ret = wlanIoctl(ifName, WLC_GET_VERSION, &val, sizeof(int));
	if (ret < 0)
	{
		return ret;
	}

	if (val != WLC_IOCTL_VERSION && 
		val != WLC_IOCTL_VERSION_LEGACY_IOTYPES)
	{
		return (-1);
	}
	
	return 0;
}

/* check wlan interface named by @ifName is work in psta mode 
 * 0 for bridge enable
 */
int wl_bridge(const char *ifName)
{
	int ret = -1;
	int psta = 0;
	int wet = 0;

	char *pstaCmdStr = "psta";
	char *wetCmdStr = "wet_tunnel";

	/* ethx */
	if ((ifName == NULL && strlen(ifName) > 2) ||
		strncmp(ifName, "eth", 3))
	{
		return (-1);
	}

	/* get psta */
	ret = wl_iovar_getint(ifName, pstaCmdStr, &psta);
	if (ret)
	{
		return ret;
	}

	/* get wet */
	ret = wl_iovar_getint(ifName, wetCmdStr, &wet);
	if (ret)
	{
		return ret;
	}

	if (psta == 0 && wet == 0)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

/* check wlan interface named by @ifName is running
 * 0 for running
 */
int wl_bssup(const char *ifName)
{	
	int ret = -1;
	int bss = 0;
	char *bssCmdStr = "bss";

	/* ethx/wlx.x */
	if ((ifName == NULL && strlen(ifName) >= 2) ||
		(strncmp(ifName, "wl", 2) && strncmp(ifName, "eth", 3)))
	{
		return (-1);
	}

	/* get bss */
	ret = wl_iovar_getint(ifName, bssCmdStr, &bss);
	if (ret)
	{
		return ret;
	}

	if (bss)
	{
		return 0;
	}
	else
	{
		return 1;
	}
}

struct bssid_addr
{
	unsigned char octet[6];
};

/* get clients assoc with @ifName, return the clients' macaddr and total num by @buf & @num 
 * 0 for success.
 */
int wl_assoclist(const char *ifName, unsigned char * buf, int bufLen)
{
	int ret = -1;
	int val = 0;

	/* ethx/wlx.x */
	if (ifName == NULL || buf == NULL)
	{
		return (-1);
	}

	/* this assignment statement is important, or ioctl would be fail */
	*(unsigned int *)buf = WLAN_MAX_STA_NUM;

	ret = wlanIoctl(ifName, WLC_GET_ASSOCLIST, (void *)buf, bufLen);
	if (ret < 0)
	{
		return ret;
	}

	return 0;
}

